import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { AudioRecorder } from 'react-audio-voice-recorder';
import { toast } from 'sonner';
import { Cake, ArrowLeft, Upload, FileText, Loader2, Mic } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const OrderPage = () => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState({
    customer_name: '',
    email: '',
    phone: '',
    delivery_address: '',
    budget: '',
    cake_size: '',
    servings: '',
    design_type: 'description',
    design_image: '',
    design_description: '',
    design_voice_note: '',
    cake_message: '',
    special_requests: ''
  });
  const [hasVoiceNote, setHasVoiceNote] = useState(false);

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleImageUpload = (e) => {
    const file = e.target.files[0];
    if (file) {
      if (file.size > 5 * 1024 * 1024) {
        toast.error('Image size should be less than 5MB');
        return;
      }
      
      const reader = new FileReader();
      reader.onloadend = () => {
        setFormData(prev => ({ ...prev, design_image: reader.result }));
        toast.success('Image uploaded successfully!');
      };
      reader.readAsDataURL(file);
    }
  };

  const handleVoiceRecording = (blob) => {
    const reader = new FileReader();
    reader.onloadend = () => {
      setFormData(prev => ({ ...prev, design_voice_note: reader.result }));
      setHasVoiceNote(true);
      toast.success('Voice note recorded successfully!');
    };
    reader.readAsDataURL(blob);
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    // Validation
    if (!formData.customer_name || !formData.email || !formData.phone || !formData.delivery_address) {
      toast.error('Please fill in all contact details');
      return;
    }
    
    if (!formData.budget || !formData.cake_size || !formData.servings) {
      toast.error('Please complete cake details');
      return;
    }
    
    if (formData.design_type === 'upload' && !formData.design_image) {
      toast.error('Please upload a design image');
      return;
    }
    
    if (formData.design_type === 'description' && !formData.design_description && !formData.design_voice_note) {
      toast.error('Please provide a design description or voice note');
      return;
    }
    
    setLoading(true);
    
    try {
      const orderData = {
        customer_name: formData.customer_name,
        email: formData.email,
        phone: formData.phone,
        delivery_address: formData.delivery_address,
        budget: parseFloat(formData.budget),
        cake_size: formData.cake_size,
        servings: parseInt(formData.servings),
        design_type: formData.design_type,
        design_image: formData.design_type === 'upload' ? formData.design_image : null,
        design_description: formData.design_type === 'description' && formData.design_description ? formData.design_description : null,
        design_voice_note: formData.design_type === 'description' && formData.design_voice_note ? formData.design_voice_note : null,
        cake_message: formData.cake_message || null,
        special_requests: formData.special_requests || null
      };
      
      const response = await axios.post(`${API}/orders`, orderData);
      
      toast.success('Order placed successfully! 🎉');
      
      // Navigate to track page with order ID
      setTimeout(() => {
        navigate(`/track?id=${response.data.id}`);
      }, 1500);
      
    } catch (error) {
      console.error('Error placing order:', error);
      toast.error(error.response?.data?.detail || 'Failed to place order. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen py-8 px-4 sm:px-6 lg:px-8">
      <div className="max-w-4xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <Button
            data-testid="back-to-home-btn"
            variant="ghost"
            onClick={() => navigate('/')}
            className="mb-4 text-pink-600 hover:text-pink-700 hover:bg-pink-50"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Home
          </Button>
          
          <div className="flex items-center space-x-3 mb-2">
            <Cake className="w-10 h-10 text-pink-500" />
            <h1 className="text-3xl sm:text-4xl font-bold text-gray-800">Order Your Custom Cake</h1>
          </div>
          <p className="text-gray-600">Fill in the details below and we'll create your perfect cake</p>
        </div>

        {/* Order Form */}
        <form onSubmit={handleSubmit} className="space-y-8">
          {/* Contact Information */}
          <div className="glass rounded-3xl p-6 sm:p-8">
            <h2 className="text-2xl font-semibold text-gray-800 mb-6">Contact Information</h2>
            <div className="grid sm:grid-cols-2 gap-6">
              <div>
                <Label htmlFor="customer_name" className="text-gray-700">Full Name *</Label>
                <Input
                  data-testid="customer-name-input"
                  id="customer_name"
                  name="customer_name"
                  value={formData.customer_name}
                  onChange={handleInputChange}
                  placeholder="John Doe"
                  required
                  className="mt-2"
                />
              </div>
              
              <div>
                <Label htmlFor="email" className="text-gray-700">Email Address *</Label>
                <Input
                  data-testid="email-input"
                  id="email"
                  name="email"
                  type="email"
                  value={formData.email}
                  onChange={handleInputChange}
                  placeholder="john@example.com"
                  required
                  className="mt-2"
                />
              </div>
              
              <div>
                <Label htmlFor="phone" className="text-gray-700">Phone Number *</Label>
                <Input
                  data-testid="phone-input"
                  id="phone"
                  name="phone"
                  value={formData.phone}
                  onChange={handleInputChange}
                  placeholder="+234 800 000 0000"
                  required
                  className="mt-2"
                />
              </div>
              
              <div>
                <Label htmlFor="delivery_address" className="text-gray-700">Delivery Address *</Label>
                <Input
                  data-testid="delivery-address-input"
                  id="delivery_address"
                  name="delivery_address"
                  value={formData.delivery_address}
                  onChange={handleInputChange}
                  placeholder="123 Main Street, Lagos"
                  required
                  className="mt-2"
                />
              </div>
            </div>
          </div>

          {/* Cake Details */}
          <div className="glass rounded-3xl p-6 sm:p-8">
            <h2 className="text-2xl font-semibold text-gray-800 mb-6">Cake Details</h2>
            <div className="grid sm:grid-cols-2 gap-6">
              <div>
                <Label htmlFor="budget" className="text-gray-700">Your Budget (₦) *</Label>
                <Input
                  data-testid="budget-input"
                  id="budget"
                  name="budget"
                  type="number"
                  min="1000"
                  step="100"
                  value={formData.budget}
                  onChange={handleInputChange}
                  placeholder="10000"
                  required
                  className="mt-2"
                />
                <p className="text-sm text-gray-500 mt-1">Minimum: ₦1,000</p>
              </div>
              
              <div>
                <Label htmlFor="cake_size" className="text-gray-700">Cake Size *</Label>
                <Select
                  value={formData.cake_size}
                  onValueChange={(value) => setFormData(prev => ({ ...prev, cake_size: value }))}
                  required
                >
                  <SelectTrigger data-testid="cake-size-select" className="mt-2">
                    <SelectValue placeholder="Select size" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="small">Small (6 inches)</SelectItem>
                    <SelectItem value="medium">Medium (8 inches)</SelectItem>
                    <SelectItem value="large">Large (10 inches)</SelectItem>
                    <SelectItem value="xlarge">Extra Large (12 inches)</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              
              <div className="sm:col-span-2">
                <Label htmlFor="servings" className="text-gray-700">Number of Servings *</Label>
                <Input
                  data-testid="servings-input"
                  id="servings"
                  name="servings"
                  type="number"
                  min="1"
                  value={formData.servings}
                  onChange={handleInputChange}
                  placeholder="20"
                  required
                  className="mt-2"
                />
              </div>
            </div>
          </div>

          {/* Design Preferences */}
          <div className="glass rounded-3xl p-6 sm:p-8">
            <h2 className="text-2xl font-semibold text-gray-800 mb-6">Design Preferences</h2>
            
            <div className="mb-6">
              <Label className="text-gray-700 mb-3 block">How would you like to share your design? *</Label>
              <RadioGroup
                value={formData.design_type}
                onValueChange={(value) => setFormData(prev => ({ ...prev, design_type: value }))}
                className="space-y-3"
              >
                <div className="flex items-center space-x-3 p-4 rounded-lg border-2 border-pink-200 hover:border-pink-400" style={{transition: 'border-color 0.3s ease'}}>
                  <RadioGroupItem data-testid="design-type-description" value="description" id="description" />
                  <Label htmlFor="description" className="flex items-center cursor-pointer flex-1">
                    <FileText className="w-5 h-5 mr-2 text-pink-500" />
                    <span>Describe your design</span>
                  </Label>
                </div>
                
                <div className="flex items-center space-x-3 p-4 rounded-lg border-2 border-pink-200 hover:border-pink-400" style={{transition: 'border-color 0.3s ease'}}>
                  <RadioGroupItem data-testid="design-type-upload" value="upload" id="upload" />
                  <Label htmlFor="upload" className="flex items-center cursor-pointer flex-1">
                    <Upload className="w-5 h-5 mr-2 text-pink-500" />
                    <span>Upload a design image</span>
                  </Label>
                </div>
              </RadioGroup>
            </div>

            {formData.design_type === 'description' && (
              <div>
                <Label htmlFor="design_description" className="text-gray-700">Describe Your Design *</Label>
                <Textarea
                  data-testid="design-description-textarea"
                  id="design_description"
                  name="design_description"
                  value={formData.design_description}
                  onChange={handleInputChange}
                  placeholder="I want a chocolate cake with strawberry frosting, decorated with fresh flowers and gold accents..."
                  rows={4}
                  required={formData.design_type === 'description'}
                  className="mt-2"
                />
              </div>
            )}

            {formData.design_type === 'upload' && (
              <div>
                <Label htmlFor="design_image" className="text-gray-700">Upload Design Image *</Label>
                <div className="mt-2">
                  <Input
                    data-testid="design-image-upload"
                    id="design_image"
                    type="file"
                    accept="image/*"
                    onChange={handleImageUpload}
                    required={formData.design_type === 'upload'}
                    className="cursor-pointer"
                  />
                  {formData.design_image && (
                    <div className="mt-4">
                      <img
                        src={formData.design_image}
                        alt="Design preview"
                        className="w-full max-w-md rounded-lg shadow-md"
                      />
                    </div>
                  )}
                </div>
                <p className="text-sm text-gray-500 mt-1">Maximum size: 5MB</p>
              </div>
            )}

            <div className="mt-6">
              <Label htmlFor="cake_message" className="text-gray-700">Cake Message/Write-up (Optional)</Label>
              <Textarea
                data-testid="cake-message-textarea"
                id="cake_message"
                name="cake_message"
                value={formData.cake_message}
                onChange={handleInputChange}
                placeholder="Happy Birthday Sarah! 🎉"
                rows={2}
                className="mt-2"
              />
            </div>

            <div className="mt-6">
              <Label htmlFor="special_requests" className="text-gray-700">Special Requests (Optional)</Label>
              <Textarea
                data-testid="special-requests-textarea"
                id="special_requests"
                name="special_requests"
                value={formData.special_requests}
                onChange={handleInputChange}
                placeholder="Any allergies, dietary restrictions, or special requests..."
                rows={3}
                className="mt-2"
              />
            </div>
          </div>

          {/* Submit Button */}
          <div className="flex justify-end space-x-4">
            <Button
              data-testid="cancel-order-btn"
              type="button"
              variant="outline"
              onClick={() => navigate('/')}
              className="border-2 border-pink-300 text-pink-600 hover:bg-pink-50 px-8 py-6 rounded-full"
            >
              Cancel
            </Button>
            <Button
              data-testid="submit-order-btn"
              type="submit"
              disabled={loading}
              className="bg-pink-500 hover:bg-pink-600 text-white px-8 py-6 rounded-full shadow-lg"
            >
              {loading ? (
                <>
                  <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                  Placing Order...
                </>
              ) : (
                'Place Order'
              )}
            </Button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default OrderPage;