import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { VoiceRecorder } from '@/components/VoiceRecorder';
import { toast } from 'sonner';
import { Cake, ArrowLeft, Upload, FileText, Loader2, Mic, Calendar } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const OrderPage = () => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [showConfirmation, setShowConfirmation] = useState(false);
  const [orderDetails, setOrderDetails] = useState(null);
  const [formData, setFormData] = useState({
    customer_name: '',
    email: '',
    phone: '',
    delivery_address: '',
    delivery_date: '',
    budget: '',
    cake_size: '',
    cake_size_other: '',
    servings: '',
    design_type: 'description',
    design_image: '',
    design_description: '',
    design_voice_note: '',
    cake_message: '',
    special_requests: '',
    packaging_requests: ''
  });
  const [hasVoiceNote, setHasVoiceNote] = useState(false);

  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const setTodayDate = () => {
    const today = new Date().toISOString().split('T')[0];
    setFormData(prev => ({ ...prev, delivery_date: today }));
  };

  const handleImageUpload = (e) => {
    const file = e.target.files[0];
    if (file) {
      if (file.size > 5 * 1024 * 1024) {
        toast.error('Image size should be less than 5MB');
        return;
      }
      
      const reader = new FileReader();
      reader.onloadend = () => {
        setFormData(prev => ({ ...prev, design_image: reader.result }));
        toast.success('Your image has been uploaded beautifully!');
      };
      reader.readAsDataURL(file);
    }
  };

  const handleVoiceRecording = (blob) => {
    const reader = new FileReader();
    reader.onloadend = () => {
      setFormData(prev => ({ ...prev, design_voice_note: reader.result }));
      setHasVoiceNote(true);
      toast.success('We heard you loud and clear! Voice note saved.');
    };
    reader.readAsDataURL(blob);
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    // Validation
    if (!formData.customer_name || !formData.phone || !formData.delivery_address) {
      toast.error('Please share your contact details with us');
      return;
    }
    
    if (!formData.delivery_date) {
      toast.error('Please let us know when you need your cake');
      return;
    }
    
    if (!formData.budget || !formData.cake_size || !formData.servings) {
      toast.error('Please tell us more about your cake');
      return;
    }
    
    if (formData.cake_size === 'other' && !formData.cake_size_other) {
      toast.error('Please specify your custom cake size');
      return;
    }
    
    // Design preferences are now optional - no validation needed
    
    setLoading(true);
    
    try {
      const orderData = {
        customer_name: formData.customer_name,
        email: formData.email,
        phone: formData.phone,
        delivery_address: formData.delivery_address,
        delivery_date: formData.delivery_date,
        budget: parseFloat(formData.budget),
        cake_size: formData.cake_size === 'other' ? `Other: ${formData.cake_size_other}` : formData.cake_size,
        servings: parseInt(formData.servings),
        design_type: formData.design_type,
        design_image: formData.design_type === 'upload' ? formData.design_image : null,
        design_description: formData.design_type === 'description' && formData.design_description ? formData.design_description : null,
        design_voice_note: formData.design_type === 'description' && formData.design_voice_note ? formData.design_voice_note : null,
        cake_message: formData.cake_message || null,
        special_requests: formData.special_requests || null,
        packaging_requests: formData.packaging_requests || null
      };
      
      const response = await axios.post(`${API}/orders`, orderData);
      
      const trackingCode = response.data.tracking_code;
      const orderId = response.data.id;
      
      // Store order details and show confirmation
      setOrderDetails({ trackingCode, orderId });
      setShowConfirmation(true);
      
      // Auto-navigate after 8 seconds
      setTimeout(() => {
        navigate(`/track?code=${trackingCode}&id=${orderId}`);
      }, 8000);
      
    } catch (error) {
      console.error('Error placing order:', error);
      toast.error(error.response?.data?.detail || 'Something went wrong. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen py-8 px-4 sm:px-6 lg:px-8 bg-white">
      <div className="max-w-4xl mx-auto">
        
        {/* Order Confirmation Modal */}
        {showConfirmation && orderDetails && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
            <div className="bg-white rounded-3xl p-8 sm:p-12 max-w-2xl w-full shadow-2xl animate-fade-in">
              <div className="text-center">
                {/* Success Icon */}
                <div className="mx-auto w-20 h-20 bg-purple-100 rounded-full flex items-center justify-center mb-6">
                  <svg className="w-12 h-12 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                  </svg>
                </div>
                
                {/* Title */}
                <h2 className="text-3xl sm:text-4xl font-bold text-gray-900 mb-4">
                  Thank you for your order! 🎉
                </h2>
                
                {/* Tracking Code */}
                <div className="bg-purple-50 border-2 border-purple-200 rounded-2xl p-4 mb-6">
                  <p className="text-sm text-gray-600 mb-1">Your Tracking Code</p>
                  <p className="text-2xl font-bold text-purple-600 font-mono tracking-wider">
                    {orderDetails.trackingCode}
                  </p>
                </div>
                
                {/* Confirmation Message */}
                <div className="text-left bg-gray-50 rounded-2xl p-6 mb-8">
                  <p className="text-gray-700 leading-relaxed mb-4">
                    We have received your order, and our expert bakers are currently reviewing the details. 
                    If there are any adjustments needed to the pricing, we will reach out with the best 
                    alternative for you.
                  </p>
                  <p className="text-gray-700 leading-relaxed">
                    You will hear from us shortly to confirm everything and finalize your order. 
                    We truly appreciate you, and we can't wait to create something special just for you!
                  </p>
                </div>
                
                {/* Action Buttons */}
                <div className="flex flex-col sm:flex-row gap-4">
                  <Button
                    onClick={() => navigate(`/track?code=${orderDetails.trackingCode}&id=${orderDetails.orderId}`)}
                    className="flex-1 bg-purple-600 hover:bg-purple-700 text-white py-6 rounded-full text-lg"
                  >
                    Track Your Order
                  </Button>
                  <Button
                    onClick={() => navigate('/')}
                    variant="outline"
                    className="flex-1 border-2 border-purple-600 text-purple-600 hover:bg-purple-50 py-6 rounded-full text-lg"
                  >
                    Back to Home
                  </Button>
                </div>
                
                <p className="text-sm text-gray-500 mt-6">
                  Redirecting to tracking page in a few seconds...
                </p>
              </div>
            </div>
          </div>
        )}
        
        {/* Header */}
        <div className="mb-8">
          <Button
            data-testid="back-to-home-btn"
            variant="ghost"
            onClick={() => navigate('/')}
            className="mb-4 text-purple-600 hover:text-purple-700 hover:bg-purple-50"
          >
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Home
          </Button>
          
          <div className="flex items-center space-x-3 mb-2">
            <Cake className="w-10 h-10 text-purple-600" />
            <h1 className="text-3xl sm:text-4xl font-bold text-gray-900">Let's Create Your Dream Cake</h1>
          </div>
          <p className="text-gray-600">We're so excited to learn about your cake fantasy. Every detail matters to us!</p>
        </div>

        {/* Order Form */}
        <form onSubmit={handleSubmit} className="space-y-8">
          {/* Contact Information */}
          <div className="bg-white border border-gray-100 rounded-3xl p-6 sm:p-8 shadow-sm">
            <h2 className="text-2xl font-semibold text-gray-900 mb-6">Tell Us About You</h2>
            <div className="grid sm:grid-cols-2 gap-6">
              <div>
                <Label htmlFor="customer_name" className="text-gray-700">Your Name *</Label>
                <Input
                  data-testid="customer-name-input"
                  id="customer_name"
                  name="customer_name"
                  value={formData.customer_name}
                  onChange={handleInputChange}
                  placeholder="What should we call you?"
                  required
                  className="mt-2"
                />
              </div>
              
              <div>
                <Label htmlFor="email" className="text-gray-700">Email Address (Optional)</Label>
                <Input
                  data-testid="email-input"
                  id="email"
                  name="email"
                  type="email"
                  value={formData.email}
                  onChange={handleInputChange}
                  placeholder="Where can we reach you?"
                  className="mt-2"
                />
              </div>
              
              <div>
                <Label htmlFor="phone" className="text-gray-700">Phone Number *</Label>
                <Input
                  data-testid="phone-input"
                  id="phone"
                  name="phone"
                  value={formData.phone}
                  onChange={handleInputChange}
                  placeholder="+234 800 000 0000"
                  required
                  className="mt-2"
                />
              </div>
              
              <div>
                <Label htmlFor="delivery_address" className="text-gray-700">Where Should We Deliver? *</Label>
                <Input
                  data-testid="delivery-address-input"
                  id="delivery_address"
                  name="delivery_address"
                  value={formData.delivery_address}
                  onChange={handleInputChange}
                  placeholder="Your address"
                  required
                  className="mt-2"
                />
              </div>
              
              <div className="sm:col-span-2">
                <Label htmlFor="delivery_date" className="text-gray-700">When do you want the order? *</Label>
                <div className="flex gap-3 mt-2">
                  <Button
                    type="button"
                    variant={formData.delivery_date === new Date().toISOString().split('T')[0] ? "default" : "outline"}
                    onClick={setTodayDate}
                    className={`${formData.delivery_date === new Date().toISOString().split('T')[0] ? 'bg-purple-600 hover:bg-purple-700 text-white' : 'border-2 border-purple-600 text-purple-600 hover:bg-purple-50'}`}
                  >
                    Today
                  </Button>
                  <Input
                    data-testid="delivery-date-input"
                    id="delivery_date"
                    name="delivery_date"
                    type="date"
                    value={formData.delivery_date}
                    onChange={handleInputChange}
                    min={new Date().toISOString().split('T')[0]}
                    required
                    className="flex-1"
                  />
                </div>
                <p className="text-sm text-gray-500 mt-1">Click "Today" for same-day delivery or select a date from the calendar</p>
              </div>
            </div>
          </div>

          {/* Cake Details */}
          <div className="bg-white border border-gray-100 rounded-3xl p-6 sm:p-8 shadow-sm">
            <h2 className="text-2xl font-semibold text-gray-900 mb-6">Your Cake Details</h2>
            <div className="grid sm:grid-cols-2 gap-6">
              <div>
                <Label htmlFor="budget" className="text-gray-700">What's Your Budget? (₦) *</Label>
                <Input
                  data-testid="budget-input"
                  id="budget"
                  name="budget"
                  type="number"
                  min="1000"
                  step="100"
                  value={formData.budget}
                  onChange={handleInputChange}
                  placeholder="10000"
                  required
                  className="mt-2"
                />
                <p className="text-sm text-gray-500 mt-1">We'll work with any amount, starting from ₦1,000</p>
              </div>
              
              <div>
                <Label htmlFor="cake_size" className="text-gray-700">How Big Should It Be? *</Label>
                <Select
                  value={formData.cake_size}
                  onValueChange={(value) => setFormData(prev => ({ ...prev, cake_size: value }))}
                  required
                >
                  <SelectTrigger data-testid="cake-size-select" className="mt-2">
                    <SelectValue placeholder="Choose a size" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="cupcakes">Cupcakes - Perfect for sharing</SelectItem>
                    <SelectItem value="small">Small (6 inches) - Intimate gatherings</SelectItem>
                    <SelectItem value="medium">Medium (8 inches) - Small parties</SelectItem>
                    <SelectItem value="large">Large (10 inches) - Big celebrations</SelectItem>
                    <SelectItem value="xlarge">Extra Large (12 inches) - Grand events</SelectItem>
                    <SelectItem value="other">Other (Specify below)</SelectItem>
                  </SelectContent>
                </Select>
                
                {/* Show input field when "Other" is selected */}
                {formData.cake_size === 'other' && (
                  <Input
                    data-testid="cake-size-other-input"
                    name="cake_size_other"
                    value={formData.cake_size_other}
                    onChange={handleInputChange}
                    placeholder="Please specify the size you need"
                    required
                    className="mt-3"
                  />
                )}
              </div>
              
              <div className="sm:col-span-2">
                <Label htmlFor="servings" className="text-gray-700">How Many People Will Enjoy This? *</Label>
                <Input
                  data-testid="servings-input"
                  id="servings"
                  name="servings"
                  type="number"
                  min="1"
                  value={formData.servings}
                  onChange={handleInputChange}
                  placeholder="Number of happy guests"
                  required
                  className="mt-2"
                />
              </div>
            </div>
          </div>

          {/* Design Preferences */}
          <div className="bg-white border border-gray-100 rounded-3xl p-6 sm:p-8 shadow-sm">
            <h2 className="text-2xl font-semibold text-gray-900 mb-2">Describe Your Dream Cake (Optional)</h2>
            <p className="text-gray-600 mb-6">Type a description, upload photos, or record a voice note - whichever works best for you!</p>
            
            <div className="mb-6">
              <Label className="text-gray-700 mb-3 block">How would you like to share your design? (Optional)</Label>
              <RadioGroup
                value={formData.design_type}
                onValueChange={(value) => setFormData(prev => ({ ...prev, design_type: value }))}
                className="space-y-3"
              >
                <div className="flex items-center space-x-3 p-4 rounded-lg border-2 border-gray-200 hover:border-purple-300" style={{transition: 'border-color 0.3s ease'}}>
                  <RadioGroupItem data-testid="design-type-description" value="description" id="description" />
                  <Label htmlFor="description" className="flex items-center cursor-pointer flex-1">
                    <FileText className="w-5 h-5 mr-2 text-purple-600" />
                    <span>Type a description or record a voice note</span>
                  </Label>
                </div>
                
                <div className="flex items-center space-x-3 p-4 rounded-lg border-2 border-gray-200 hover:border-purple-300" style={{transition: 'border-color 0.3s ease'}}>
                  <RadioGroupItem data-testid="design-type-upload" value="upload" id="upload" />
                  <Label htmlFor="upload" className="flex items-center cursor-pointer flex-1">
                    <Upload className="w-5 h-5 mr-2 text-purple-600" />
                    <span>Upload a photo or design</span>
                  </Label>
                </div>
              </RadioGroup>
            </div>

            {formData.design_type === 'description' && (
              <div className="space-y-6">
                <div>
                  <Label htmlFor="design_description" className="text-gray-700">Describe Your Perfect Cake (Optional)</Label>
                  <Textarea
                    data-testid="design-description-textarea"
                    id="design_description"
                    name="design_description"
                    value={formData.design_description}
                    onChange={handleInputChange}
                    placeholder="Tell us what you envision... e.g., chocolate layers with strawberry frosting, decorated with fresh roses and gold accents..."
                    rows={4}
                    className="mt-2"
                  />
                </div>

                <div className="p-6 rounded-2xl bg-purple-50 border-2 border-dashed border-purple-200">
                  <div className="flex items-center space-x-2 mb-3">
                    <Mic className="w-5 h-5 text-purple-600" />
                    <Label className="text-gray-700 font-semibold">Or Record a Voice Note (Optional)</Label>
                  </div>
                  <p className="text-sm text-gray-600 mb-4">Sometimes words flow better when spoken. Record your cake fantasy and we'll listen!</p>
                  <div className="flex justify-center" data-testid="voice-recorder">
                    <VoiceRecorder onRecordingComplete={handleVoiceRecording} />
                  </div>
                  {hasVoiceNote && (
                    <p className="text-center mt-3 text-sm font-medium text-purple-600">
                      ✓ Voice note received! We can't wait to hear your ideas.
                    </p>
                  )}
                </div>
              </div>
            )}

            {formData.design_type === 'upload' && (
              <div>
                <Label htmlFor="design_image" className="text-gray-700">Upload Your Inspiration (Optional)</Label>
                <div className="mt-2">
                  <Input
                    data-testid="design-image-upload"
                    id="design_image"
                    type="file"
                    accept="image/*"
                    onChange={handleImageUpload}
                    className="cursor-pointer"
                  />
                  {formData.design_image && (
                    <div className="mt-4">
                      <p className="text-sm text-gray-600 mb-2">Your inspiration looks beautiful!</p>
                      <img
                        src={formData.design_image}
                        alt="Design preview"
                        className="w-full max-w-md rounded-lg shadow-md"
                      />
                    </div>
                  )}
                </div>
                <p className="text-sm text-gray-500 mt-1">Maximum size: 5MB</p>
              </div>
            )}

            <div className="mt-6">
              <Label htmlFor="cake_message" className="text-gray-700">What Message Should We Write? (Optional)</Label>
              <Textarea
                data-testid="cake-message-textarea"
                id="cake_message"
                name="cake_message"
                value={formData.cake_message}
                onChange={handleInputChange}
                placeholder="Happy Birthday Sarah! May your day be as sweet as this cake 🎉"
                rows={2}
                className="mt-2"
              />
            </div>

            <div className="mt-6">
              <Label htmlFor="special_requests" className="text-gray-700">Anything Else We Should Know? (Optional)</Label>
              <Textarea
                data-testid="special-requests-textarea"
                id="special_requests"
                name="special_requests"
                value={formData.special_requests}
                onChange={handleInputChange}
                placeholder="Allergies, dietary needs, delivery time preferences... we're all ears!"
                rows={3}
                className="mt-2"
              />
            </div>

            <div className="mt-6">
              <Label htmlFor="packaging_requests" className="text-gray-700">Special Packaging Requests (Optional)</Label>
              <Textarea
                data-testid="packaging-requests-textarea"
                id="packaging_requests"
                name="packaging_requests"
                value={formData.packaging_requests || ''}
                onChange={handleInputChange}
                placeholder="Need special gift wrapping? Custom box? Delivery instructions? Let us know your packaging preferences..."
                rows={3}
                className="mt-2"
              />
            </div>
          </div>

          {/* Submit Button */}
          <div className="flex justify-end space-x-4">
            <Button
              data-testid="cancel-order-btn"
              type="button"
              variant="outline"
              onClick={() => navigate('/')}
              className="border-2 border-purple-600 text-purple-600 hover:bg-purple-50 px-8 py-6 rounded-full"
            >
              Not Yet
            </Button>
            <Button
              data-testid="submit-order-btn"
              type="submit"
              disabled={loading}
              className="bg-purple-600 hover:bg-purple-700 text-white px-8 py-6 rounded-full shadow-lg"
            >
              {loading ? (
                <>
                  <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                  Creating Your Magic...
                </>
              ) : (
                'Yes, Create My Dream Cake!'
              )}
            </Button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default OrderPage;
