import { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { toast } from 'sonner';
import { Cake, ArrowLeft, RefreshCw, LogOut } from 'lucide-react';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const ADMIN_CREDENTIALS = {
  username: 'hello@cakecamp',
  password: 'Fortune2025'
};

const AdminPage = () => {
  const navigate = useNavigate();
  const [orders, setOrders] = useState([]);
  const [loading, setLoading] = useState(false);
  const [filter, setFilter] = useState('all');
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [loginForm, setLoginForm] = useState({ username: '', password: '' });

  useEffect(() => {
    // Check if user is already authenticated
    const authStatus = sessionStorage.getItem('adminAuthenticated');
    if (authStatus === 'true') {
      setIsAuthenticated(true);
      fetchOrders();
    }
  }, []);

  const handleLogin = (e) => {
    e.preventDefault();
    if (loginForm.username === ADMIN_CREDENTIALS.username && 
        loginForm.password === ADMIN_CREDENTIALS.password) {
      setIsAuthenticated(true);
      sessionStorage.setItem('adminAuthenticated', 'true');
      toast.success('Welcome to CakeCamp Admin!');
      fetchOrders();
    } else {
      toast.error('Invalid username or password');
    }
  };

  const handleLogout = () => {
    setIsAuthenticated(false);
    sessionStorage.removeItem('adminAuthenticated');
    toast.success('Logged out successfully');
  };

  const updateTrackingStage = async (orderId, stageIndex) => {
    try {
      await axios.patch(`${API}/orders/${orderId}/tracking?stage_index=${stageIndex}`);
      toast.success('Tracking stage updated!');
      fetchOrders(); // Refresh orders
    } catch (error) {
      console.error('Error updating tracking stage:', error);
      toast.error('Failed to update tracking stage');
    }
  };

  const fetchOrders = async () => {
    setLoading(true);
    try {
      const response = await axios.get(`${API}/orders`);
      setOrders(response.data);
    } catch (error) {
      console.error('Error fetching orders:', error);
      toast.error('Failed to load orders');
    } finally {
      setLoading(false);
    }
  };

  const updateOrderStatus = async (orderId, newStatus) => {
    try {
      await axios.patch(`${API}/orders/${orderId}`, { status: newStatus });
      toast.success('Order status updated!');
      fetchOrders();
    } catch (error) {
      console.error('Error updating order:', error);
      toast.error('Failed to update order status');
    }
  };

  const filteredOrders = orders.filter(order => {
    if (filter === 'all') return true;
    return order.status === filter;
  });

  const getStatusColor = (status) => {
    switch (status) {
      case 'pending': return 'bg-yellow-100 text-yellow-800';
      case 'confirmed': return 'bg-blue-100 text-blue-800';
      case 'in_progress': return 'bg-purple-100 text-purple-800';
      case 'completed': return 'bg-green-100 text-green-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  // Login Form
  if (!isAuthenticated) {
    return (
      <div className="min-h-screen flex items-center justify-center px-4 bg-gradient-to-br from-purple-50 to-white">
        <div className="max-w-md w-full">
          <div className="text-center mb-8">
            <Cake className="w-16 h-16 text-purple-600 mx-auto mb-4" />
            <h1 className="text-3xl font-bold text-gray-900 mb-2">CakeCamp Admin</h1>
            <p className="text-gray-600">Sign in to access the dashboard</p>
          </div>

          <form onSubmit={handleLogin} className="bg-white rounded-3xl p-8 shadow-lg border border-gray-100">
            <div className="space-y-6">
              <div>
                <Label htmlFor="username" className="text-gray-700">Username</Label>
                <Input
                  id="username"
                  type="text"
                  value={loginForm.username}
                  onChange={(e) => setLoginForm({ ...loginForm, username: e.target.value })}
                  placeholder="Enter username"
                  required
                  className="mt-2"
                />
              </div>

              <div>
                <Label htmlFor="password" className="text-gray-700">Password</Label>
                <Input
                  id="password"
                  type="password"
                  value={loginForm.password}
                  onChange={(e) => setLoginForm({ ...loginForm, password: e.target.value })}
                  placeholder="Enter password"
                  required
                  className="mt-2"
                />
              </div>

              <Button
                type="submit"
                className="w-full bg-purple-600 hover:bg-purple-700 text-white py-6 rounded-full"
              >
                Sign In
              </Button>
            </div>
          </form>

          <div className="text-center mt-6">
            <Button
              variant="ghost"
              onClick={() => navigate('/')}
              className="text-purple-600 hover:text-purple-700"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Home
            </Button>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen py-8 px-4 sm:px-6 lg:px-8 bg-white">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <div className="flex justify-between items-start mb-4">
            <Button
              data-testid="back-to-home-btn"
              variant="ghost"
              onClick={() => navigate('/')}
              className="text-purple-600 hover:text-purple-700 hover:bg-purple-50"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back to Home
            </Button>
            
            <Button
              onClick={handleLogout}
              variant="ghost"
              className="text-red-600 hover:text-red-700 hover:bg-red-50"
            >
              <LogOut className="w-4 h-4 mr-2" />
              Logout
            </Button>
          </div>
          
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <Cake className="w-10 h-10 text-purple-600" />
              <div>
                <h1 className="text-3xl sm:text-4xl font-bold text-gray-900">Admin Dashboard</h1>
                <p className="text-gray-600">Manage all cake orders</p>
              </div>
            </div>
            <Button
              data-testid="refresh-orders-btn"
              onClick={fetchOrders}
              disabled={loading}
              className="bg-purple-600 hover:bg-purple-700 text-white rounded-full"
            >
              <RefreshCw className={`w-5 h-5 mr-2 ${loading ? 'animate-spin' : ''}`} />
              Refresh
            </Button>
          </div>
        </div>

        {/* Filter */}
        <div className="bg-white border border-gray-100 rounded-3xl p-6 mb-8 shadow-sm">
          <div className="flex items-center space-x-4">
            <span className="font-semibold text-gray-700">Filter by Status:</span>
            <Select value={filter} onValueChange={setFilter}>
              <SelectTrigger data-testid="status-filter-select" className="w-48">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Orders</SelectItem>
                <SelectItem value="pending">Pending</SelectItem>
                <SelectItem value="confirmed">Confirmed</SelectItem>
                <SelectItem value="in_progress">In Progress</SelectItem>
                <SelectItem value="completed">Completed</SelectItem>
                <SelectItem value="cancelled">Cancelled</SelectItem>
              </SelectContent>
            </Select>
            <span className="text-gray-600">Total: {filteredOrders.length}</span>
          </div>
        </div>

        {/* Orders List */}
        <div className="space-y-6">
          {filteredOrders.length === 0 ? (
            <div className="bg-white border border-gray-100 rounded-3xl p-12 text-center shadow-sm">
              <p className="text-gray-600 text-lg">No orders found</p>
            </div>
          ) : (
            filteredOrders.map((order) => (
              <div key={order.id} data-testid={`order-card-${order.id}`} className="bg-white border border-gray-100 rounded-3xl p-6 sm:p-8 shadow-sm">
                <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6">
                  <div>
                    <h3 className="text-xl font-semibold text-gray-800 mb-1">{order.customer_name}</h3>
                    <p className="text-sm text-gray-600">Order ID: {order.id}</p>
                    <p className="text-sm font-mono font-bold text-purple-600 mt-1">
                      Tracking: {order.tracking_code || 'N/A'}
                    </p>
                  </div>
                  <div className="mt-4 sm:mt-0">
                    <span className={`px-4 py-2 rounded-full text-sm font-semibold ${getStatusColor(order.status)}`}>
                      {order.status.replace('_', ' ').toUpperCase()}
                    </span>
                  </div>
                </div>

                <div className="grid sm:grid-cols-2 lg:grid-cols-3 gap-4 mb-6">
                  <div>
                    <p className="text-sm text-gray-600">Budget</p>
                    <p className="font-semibold text-gray-800">₦{order.budget.toLocaleString()}</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Size & Servings</p>
                    <p className="font-semibold text-gray-800 capitalize">{order.cake_size} ({order.servings} people)</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-600">Contact</p>
                    <p className="font-semibold text-gray-800">{order.phone}</p>
                  </div>
                  <div className="sm:col-span-2">
                    <p className="text-sm text-gray-600">Delivery Address</p>
                    <p className="font-semibold text-gray-800">{order.delivery_address}</p>
                  </div>
                  {order.cake_message && (
                    <div className="sm:col-span-2 lg:col-span-3">
                      <p className="text-sm text-gray-600">Cake Message</p>
                      <p className="font-semibold text-gray-800">{order.cake_message}</p>
                    </div>
                  )}
                  {order.design_description && (
                    <div className="sm:col-span-2 lg:col-span-3">
                      <p className="text-sm text-gray-600">Design Description</p>
                      <p className="font-semibold text-gray-800">{order.design_description}</p>
                    </div>
                  )}
                </div>

                {/* Tracking Stages */}
                {order.tracking_stages && (
                  <div className="mt-6 border-t pt-6">
                    <h4 className="font-semibold text-gray-800 mb-4">Order Tracking Stages</h4>
                    <div className="grid sm:grid-cols-2 lg:grid-cols-3 gap-3">
                      {order.tracking_stages.map((stage, index) => (
                        <Button
                          key={index}
                          onClick={() => updateTrackingStage(order.id, index)}
                          disabled={stage.completed}
                          size="sm"
                          variant="outline"
                          className={`${
                            stage.completed 
                              ? 'bg-purple-100 border-purple-300 text-purple-800' 
                              : 'border-gray-300 text-gray-700 hover:bg-purple-50'
                          } flex items-center justify-between`}
                        >
                          <span className="truncate">{stage.stage}</span>
                          {stage.completed && <span className="ml-2">✓</span>}
                        </Button>
                      ))}
                    </div>
                  </div>
                )}

                <div className="flex flex-wrap gap-2 mt-6">
                  <Button
                    data-testid={`update-status-pending-${order.id}`}
                    onClick={() => updateOrderStatus(order.id, 'pending')}
                    disabled={order.status === 'pending'}
                    size="sm"
                    variant="outline"
                    className="border-yellow-300 text-yellow-700 hover:bg-yellow-50"
                  >
                    Pending
                  </Button>
                  <Button
                    data-testid={`update-status-confirmed-${order.id}`}
                    onClick={() => updateOrderStatus(order.id, 'confirmed')}
                    disabled={order.status === 'confirmed'}
                    size="sm"
                    variant="outline"
                    className="border-blue-300 text-blue-700 hover:bg-blue-50"
                  >
                    Confirm
                  </Button>
                  <Button
                    data-testid={`update-status-in-progress-${order.id}`}
                    onClick={() => updateOrderStatus(order.id, 'in_progress')}
                    disabled={order.status === 'in_progress'}
                    size="sm"
                    variant="outline"
                    className="border-purple-300 text-purple-700 hover:bg-purple-50"
                  >
                    In Progress
                  </Button>
                  <Button
                    data-testid={`update-status-completed-${order.id}`}
                    onClick={() => updateOrderStatus(order.id, 'completed')}
                    disabled={order.status === 'completed'}
                    size="sm"
                    variant="outline"
                    className="border-green-300 text-green-700 hover:bg-green-50"
                  >
                    Complete
                  </Button>
                  <Button
                    data-testid={`update-status-cancelled-${order.id}`}
                    onClick={() => updateOrderStatus(order.id, 'cancelled')}
                    disabled={order.status === 'cancelled'}
                    size="sm"
                    variant="outline"
                    className="border-red-300 text-red-700 hover:bg-red-50"
                  >
                    Cancel
                  </Button>
                </div>
              </div>
            ))
          )}
        </div>
      </div>
    </div>
  );
};

export default AdminPage;